package com.nyazo.android

import android.Manifest
import android.content.Intent
import android.content.pm.PackageManager
import android.os.Build
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.*
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView

class MainActivity : AppCompatActivity() {
    private lateinit var configManager: ConfigManager
    private lateinit var historyManager: HistoryManager
    private lateinit var uploadManager: UploadManager
    private lateinit var serverUrlInput: EditText
    private lateinit var apiKeyInput: EditText
    private lateinit var deletePasswordInput: EditText
    private lateinit var serviceSwitch: Switch
    private lateinit var saveButton: Button
    private lateinit var historyRecyclerView: RecyclerView
    private lateinit var clearHistoryButton: Button
    private lateinit var adapter: HistoryAdapter
    
    companion object {
        private const val PERMISSION_REQUEST_CODE = 1001
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.activity_main)
        
        configManager = ConfigManager(this)
        historyManager = HistoryManager(this)
        uploadManager = UploadManager(this)
        
        initViews()
        loadConfig()
        loadHistory()
        checkAndRequestPermissions()
    }
    
    override fun onResume() {
        super.onResume()
        loadHistory() // Обновляем историю при возврате
    }

    private fun initViews() {
        serverUrlInput = findViewById(R.id.serverUrlInput)
        apiKeyInput = findViewById(R.id.apiKeyInput)
        deletePasswordInput = findViewById(R.id.deletePasswordInput)
        serviceSwitch = findViewById(R.id.serviceSwitch)
        saveButton = findViewById(R.id.saveButton)
        historyRecyclerView = findViewById(R.id.historyRecyclerView)
        clearHistoryButton = findViewById(R.id.clearHistoryButton)
        
        historyRecyclerView.layoutManager = LinearLayoutManager(this)
        adapter = HistoryAdapter(emptyList()) { item ->
            showDeleteDialog(item)
        }
        historyRecyclerView.adapter = adapter
        
        saveButton.setOnClickListener {
            saveConfig()
        }
        
        clearHistoryButton.setOnClickListener {
            historyManager.clearHistory()
            loadHistory()
            Toast.makeText(this, "История очищена", Toast.LENGTH_SHORT).show()
        }
        
        serviceSwitch.setOnCheckedChangeListener { _, isChecked ->
            if (isChecked) {
                startScreenshotService()
            } else {
                stopScreenshotService()
            }
        }
    }
    
    private fun loadHistory() {
        val history = historyManager.getHistory()
        adapter.updateData(history)
    }
    
    private fun showDeleteDialog(item: ScreenshotItem) {
        AlertDialog.Builder(this)
            .setTitle("Удалить скриншот?")
            .setMessage(item.fileName)
            .setPositiveButton("Удалить") { _, _ ->
                deleteScreenshot(item)
            }
            .setNegativeButton("Отмена", null)
            .show()
    }
    
    private fun deleteScreenshot(item: ScreenshotItem) {
        Thread {
            val config = configManager.loadConfig()
            val success = uploadManager.deleteScreenshot(item.url, config.deletePassword)
            
            runOnUiThread {
                if (success) {
                    historyManager.removeScreenshot(item.url)
                    loadHistory()
                    Toast.makeText(this, "Скриншот удален", Toast.LENGTH_SHORT).show()
                } else {
                    Toast.makeText(this, "Ошибка удаления", Toast.LENGTH_SHORT).show()
                }
            }
        }.start()
    }

    private fun loadConfig() {
        val config = configManager.loadConfig()
        serverUrlInput.setText(config.serverUrl)
        apiKeyInput.setText(config.apiKey)
        deletePasswordInput.setText(config.deletePassword)
    }

    private fun saveConfig() {
        val config = Config(
            serverUrl = serverUrlInput.text.toString(),
            apiKey = apiKeyInput.text.toString(),
            deletePassword = deletePasswordInput.text.toString()
        )
        configManager.saveConfig(config)
        Toast.makeText(this, "Конфигурация сохранена", Toast.LENGTH_SHORT).show()
    }

    private fun checkAndRequestPermissions() {
        val permissions = mutableListOf<String>()
        
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            if (ContextCompat.checkSelfPermission(this, Manifest.permission.READ_MEDIA_IMAGES) 
                != PackageManager.PERMISSION_GRANTED) {
                permissions.add(Manifest.permission.READ_MEDIA_IMAGES)
            }
            if (ContextCompat.checkSelfPermission(this, Manifest.permission.POST_NOTIFICATIONS) 
                != PackageManager.PERMISSION_GRANTED) {
                permissions.add(Manifest.permission.POST_NOTIFICATIONS)
            }
        } else {
            if (ContextCompat.checkSelfPermission(this, Manifest.permission.READ_EXTERNAL_STORAGE) 
                != PackageManager.PERMISSION_GRANTED) {
                permissions.add(Manifest.permission.READ_EXTERNAL_STORAGE)
            }
        }
        
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.UPSIDE_DOWN_CAKE) {
            if (ContextCompat.checkSelfPermission(this, Manifest.permission.DETECT_SCREEN_CAPTURE) 
                != PackageManager.PERMISSION_GRANTED) {
                permissions.add(Manifest.permission.DETECT_SCREEN_CAPTURE)
            }
        }
        
        if (permissions.isNotEmpty()) {
            ActivityCompat.requestPermissions(
                this,
                permissions.toTypedArray(),
                PERMISSION_REQUEST_CODE
            )
        }
    }

    private fun startScreenshotService() {
        val intent = Intent(this, ScreenshotDetectionService::class.java)
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            startForegroundService(intent)
        } else {
            startService(intent)
        }
        Toast.makeText(this, "Сервис запущен", Toast.LENGTH_SHORT).show()
    }

    private fun stopScreenshotService() {
        val intent = Intent(this, ScreenshotDetectionService::class.java)
        stopService(intent)
        Toast.makeText(this, "Сервис остановлен", Toast.LENGTH_SHORT).show()
    }

    override fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>,
        grantResults: IntArray
    ) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults)
        if (requestCode == PERMISSION_REQUEST_CODE) {
            if (grantResults.all { it == PackageManager.PERMISSION_GRANTED }) {
                Toast.makeText(this, "Разрешения получены", Toast.LENGTH_SHORT).show()
            } else {
                Toast.makeText(this, "Необходимы все разрешения", Toast.LENGTH_LONG).show()
            }
        }
    }
}

class HistoryAdapter(
    private var items: List<ScreenshotItem>,
    private val onDelete: (ScreenshotItem) -> Unit
) : RecyclerView.Adapter<HistoryAdapter.ViewHolder>() {

    class ViewHolder(view: View) : RecyclerView.ViewHolder(view) {
        val fileName: TextView = view.findViewById(R.id.fileName)
        val url: TextView = view.findViewById(R.id.url)
        val deleteButton: Button = view.findViewById(R.id.deleteButton)
        val copyButton: Button = view.findViewById(R.id.copyButton)
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ViewHolder {
        val view = LayoutInflater.from(parent.context)
            .inflate(R.layout.item_history, parent, false)
        return ViewHolder(view)
    }

    override fun onBindViewHolder(holder: ViewHolder, position: Int) {
        val item = items[position]
        holder.fileName.text = item.fileName
        holder.url.text = item.url
        
        holder.deleteButton.setOnClickListener {
            onDelete(item)
        }
        
        holder.copyButton.setOnClickListener {
            val clipboard = holder.itemView.context.getSystemService(android.content.Context.CLIPBOARD_SERVICE) 
                as android.content.ClipboardManager
            val clip = android.content.ClipData.newPlainText("URL", item.url)
            clipboard.setPrimaryClip(clip)
            Toast.makeText(holder.itemView.context, "Ссылка скопирована", Toast.LENGTH_SHORT).show()
        }
    }

    override fun getItemCount() = items.size

    fun updateData(newItems: List<ScreenshotItem>) {
        items = newItems
        notifyDataSetChanged()
    }
}
