package com.nyazo.android
import java.io.File

import android.app.*
import android.content.ClipData
import android.content.ClipboardManager
import android.content.Context
import android.content.Intent
import android.database.ContentObserver
import android.database.Cursor
import android.net.Uri
import android.os.Build
import android.os.Handler
import android.os.IBinder
import android.os.Looper
import android.provider.MediaStore
import android.widget.Toast
import androidx.core.app.NotificationCompat

class ScreenshotDetectionService : Service() {
    private val handler = Handler(Looper.getMainLooper())
    private var screenshotObserver: ContentObserver? = null
    private lateinit var uploadManager: UploadManager
    private lateinit var historyManager: HistoryManager
    private val processedPaths = mutableSetOf<String>()
    private var lastCheckTime: Long = 0
    
    private val pollingRunnable = object : Runnable {
        override fun run() {
            checkForNewScreenshots()
            handler.postDelayed(this, 2000)
        }
    }
    
    companion object {
        private const val NOTIFICATION_ID = 1001
        private const val CHANNEL_ID = "screenshot_upload_channel"
    }

    override fun onCreate() {
        super.onCreate()
        uploadManager = UploadManager(this)
        historyManager = HistoryManager(this)
        createNotificationChannel()
        startForeground(NOTIFICATION_ID, createNotification("Мониторинг активен"))
        startScreenshotObserver()
        lastCheckTime = System.currentTimeMillis() - 60000
        handler.postDelayed(pollingRunnable, 2000)
    }

    private fun startScreenshotObserver() {
        screenshotObserver = object : ContentObserver(Handler(Looper.getMainLooper())) {
            override fun onChange(selfChange: Boolean, uri: Uri?) {
                super.onChange(selfChange, uri)
                uri?.let { 
                    handler.postDelayed({ handleUri(it) }, 200) // Было 500, теперь 200
                }
            }
        }

        contentResolver.registerContentObserver(
            MediaStore.Images.Media.EXTERNAL_CONTENT_URI,
            true,
            screenshotObserver!!
        )
    }

    private fun handleUri(uri: Uri) {
        Thread {
            try {
                val path = getImagePath(uri)
                if (path != null && !processedPaths.contains(path) && isScreenshot(path)) {
                    processScreenshot(path)
                }
            } catch (e: Exception) {
                // Ignore
            }
        }.start()
    }

    private fun checkForNewScreenshots() {
        Thread {
            try {
                val projection = arrayOf(
                    MediaStore.Images.Media.DATA,
                    MediaStore.Images.Media.DATE_ADDED
                )
                
                val selection = "${MediaStore.Images.Media.DATE_ADDED} > ?"
                val selectionArgs = arrayOf((lastCheckTime / 1000).toString())
                
                val cursor: Cursor? = contentResolver.query(
                    MediaStore.Images.Media.EXTERNAL_CONTENT_URI,
                    projection,
                    selection,
                    selectionArgs,
                    "${MediaStore.Images.Media.DATE_ADDED} DESC"
                )
                
                cursor?.use {
                    val dataColumn = it.getColumnIndexOrThrow(MediaStore.Images.Media.DATA)
                    val dateColumn = it.getColumnIndexOrThrow(MediaStore.Images.Media.DATE_ADDED)
                    
                    while (it.moveToNext()) {
                        val path = it.getString(dataColumn)
                        val dateAdded = it.getLong(dateColumn) * 1000
                        
                        if (isScreenshot(path) && !processedPaths.contains(path)) {
                            processScreenshot(path)
                        }
                        
                        if (dateAdded > lastCheckTime) {
                            lastCheckTime = dateAdded
                        }
                    }
                }
            } catch (e: Exception) {
                // Ignore
            }
        }.start()
    }

    private fun processScreenshot(path: String) {
        if (processedPaths.contains(path)) {
            return
        }
        
        processedPaths.add(path)
        
        if (processedPaths.size > 50) {
            val toRemove = processedPaths.take(processedPaths.size - 50)
            processedPaths.removeAll(toRemove.toSet())
        }
        
        uploadWithRetry(path, 0)
    }

    private fun uploadWithRetry(path: String, attempt: Int) {
        Thread {
            try {
                val file = File(path)
                
                if (!file.exists() || file.length() == 0L || !file.canRead()) {
                    if (attempt < 3) {
                        Thread.sleep(500) // Было 1000, теперь 500
                        uploadWithRetry(path, attempt + 1)
                    }
                    return@Thread
                }
                
                handler.post {
                    updateNotification("Загрузка...")
                }
                
                val url = uploadManager.uploadScreenshotSync(path)
                
                if (!url.isNullOrEmpty()) {
                    copyToClipboard(url)
                    historyManager.addScreenshot(url)
                    handler.post {
                        Toast.makeText(this, "Ссылка скопирована", Toast.LENGTH_SHORT).show()
                        updateNotification("Загружен")
                    }
                } else {
                    if (attempt < 2) {
                        Thread.sleep(500) // Было 1000, теперь 500
                        uploadWithRetry(path, attempt + 1)
                    } else {
                        handler.post {
                            Toast.makeText(this, "Ошибка загрузки", Toast.LENGTH_SHORT).show()
                            updateNotification("Ошибка")
                        }
                    }
                }
                
                handler.postDelayed({
                    updateNotification("Мониторинг активен")
                }, 1000) // Было 2000, теперь 1000
                
            } catch (e: Exception) {
                if (attempt < 2) {
                    Thread.sleep(500) // Было 1000, теперь 500
                    uploadWithRetry(path, attempt + 1)
                }
            }
        }.start()
    }

    private fun getImagePath(uri: Uri): String? {
        val projection = arrayOf(MediaStore.Images.Media.DATA)
        contentResolver.query(uri, projection, null, null, null)?.use { cursor ->
            if (cursor.moveToFirst()) {
                val columnIndex = cursor.getColumnIndexOrThrow(MediaStore.Images.Media.DATA)
                return cursor.getString(columnIndex)
            }
        }
        return null
    }

    private fun isScreenshot(path: String): Boolean {
        val fileName = File(path).name.lowercase()
        return fileName.startsWith("screenshot")
    }

    private fun copyToClipboard(text: String) {
        val clipboard = getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
        val clip = ClipData.newPlainText("Nyazo URL", text)
        clipboard.setPrimaryClip(clip)
    }

    private fun createNotificationChannel() {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            val channel = NotificationChannel(
                CHANNEL_ID,
                "Загрузка скриншотов",
                NotificationManager.IMPORTANCE_MIN
            ).apply {
                description = "Мониторинг и загрузка скриншотов"
                setShowBadge(false)
                enableLights(false)
                enableVibration(false)
                setSound(null, null)
            }
            
            val notificationManager = getSystemService(NotificationManager::class.java)
            notificationManager.createNotificationChannel(channel)
        }
    }

    private fun createNotification(text: String): Notification {
        val intent = Intent(this, MainActivity::class.java)
        val pendingIntent = PendingIntent.getActivity(
            this, 0, intent,
            PendingIntent.FLAG_IMMUTABLE
        )

        return NotificationCompat.Builder(this, CHANNEL_ID)
            .setContentTitle("Nyazo")
            .setContentText(text)
            .setSmallIcon(R.drawable.ic_notification)
            .setContentIntent(pendingIntent)
            .setPriority(NotificationCompat.PRIORITY_MIN)
            .setOngoing(true)
            .setSilent(true)
            .setShowWhen(false)
            .setVisibility(NotificationCompat.VISIBILITY_SECRET)
            .build()
    }

    private fun updateNotification(text: String) {
        try {
            val notificationManager = getSystemService(NotificationManager::class.java)
            notificationManager.notify(NOTIFICATION_ID, createNotification(text))
        } catch (e: Exception) {
            // Ignore
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        screenshotObserver?.let {
            contentResolver.unregisterContentObserver(it)
        }
        handler.removeCallbacks(pollingRunnable)
    }

    override fun onBind(intent: Intent?): IBinder? = null
}
