package com.nyazo.android

import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import okhttp3.*
import okhttp3.MediaType.Companion.toMediaType
import okhttp3.RequestBody.Companion.asRequestBody
import java.io.File
import java.io.FileOutputStream
import okhttp3.FormBody


class UploadManager(private val context: Context) {
    private val client = OkHttpClient()
    private val configManager = ConfigManager(context)

    fun uploadScreenshotSync(imagePath: String): String? {
        return try {
            val config = configManager.loadConfig()
            val file = File(imagePath)
            
            if (!file.exists() || file.length() == 0L) {
                return null
            }

            // Конвертируем в PNG если это JPG
            val pngFile = convertToPng(file)

            val requestBody = MultipartBody.Builder()
                .setType(MultipartBody.FORM)
                .addFormDataPart(
                    "file",
                    pngFile.name,
                    pngFile.asRequestBody("image/png".toMediaType())
                )
                .build()
            
            val request = Request.Builder()
                .url(config.serverUrl)
                .apply {
                    if (config.apiKey.isNotEmpty()) {
                        addHeader("Authorization", "Bearer ${config.apiKey}")
                    }
                }
                .post(requestBody)
                .build()

            client.newCall(request).execute().use { response ->
                if (!response.isSuccessful) {
                    return null
                }

                val responseBody = response.body?.string()?.trim() ?: ""
                
                if (responseBody.startsWith("http://") || responseBody.startsWith("https://")) {
                    return responseBody
                }
                
                return null
            }
        } catch (e: Exception) {
            return null
        }
    }

    fun deleteScreenshot(url: String, password: String): Boolean {
        return try {
            // Из https://dayforum.icu/fwMonKM.png делаем https://dayforum.icu/del/fwMonKM.png
            val fileName = url.substringAfterLast("/")
            val baseUrl = url.substringBeforeLast("/")
            val deleteUrl = "$baseUrl/del/$fileName"

            val formBody = FormBody.Builder()
            .add("password", password)
            .build()

            val request = Request.Builder()
            .url(deleteUrl)
            .post(formBody)
            .build()

            client.newCall(request).execute().use { response ->
                response.isSuccessful
            }
        } catch (e: Exception) {
            false
        }
    }


    private fun convertToPng(file: File): File {
        return try {
            // Если уже PNG - возвращаем как есть
            if (file.extension.lowercase() == "png") {
                return file
            }
            
            // Конвертируем JPG в PNG
            val bitmap = BitmapFactory.decodeFile(file.absolutePath) ?: return file
            
            val pngFile = File(context.cacheDir, "upload_${System.currentTimeMillis()}.png")
            
            FileOutputStream(pngFile).use { out ->
                bitmap.compress(Bitmap.CompressFormat.PNG, 100, out)
            }
            
            bitmap.recycle()
            pngFile
        } catch (e: Exception) {
            file
        }
    }
}
